<?php

namespace App\Http\Controllers;

use App\Helpers\NotificationHelper;
use App\Models\Task;
use App\Models\TaskApproval;
use App\Models\TaskLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ApprovalController extends Controller
{
    // List tasks pending approval
    public function index()
    {
        $tasks = Task::with(['assignedUser', 'createdByUser'])
            ->where('status', 'under_review')
            ->orderBy('updated_at', 'desc')
            ->paginate(15);

        return view('approvals.index', compact('tasks'));
    }

    // Approve a task
    public function approve(Request $request, Task $task)
    {
        $request->validate([
            'comment' => 'nullable|string|max:500',
        ]);

        if ($task->status !== 'under_review') {
            return back()->with('error', 'Only tasks under review can be approved.');
        }

        $reviewer = Auth::user();

        // 1. Mark original task as approved
        $task->update(['status' => 'approved']);

        TaskApproval::create([
            'task_id'     => $task->id,
            'reviewed_by' => $reviewer->id,
            'status'      => 'approved',
            'comment'     => $request->comment,
        ]);

        TaskLog::create([
            'task_id' => $task->id,
            'user_id' => $reviewer->id,
            'action'  => 'approved',
            'note'    => $request->comment ?? 'Task approved by ' . $reviewer->name,
        ]);

        // 2. Auto-create a marketing task ONLY for tasks that belong to another user
        //    (not when marketing is approving their own marketing processing task)
        $isMarketingProcessingTask = $task->parent_task_id !== null;
        $isOwnTask                 = $task->assigned_to === $reviewer->id;

        if (!$isMarketingProcessingTask && !$isOwnTask) {
            $marketingTask = Task::create([
                'title'          => '[Marketing] ' . $task->title,
                'description'    => 'Marketing processing task for: "' . $task->title . '"' . PHP_EOL .
                                    'Original task completed by: ' . $task->assignedUser->name . PHP_EOL .
                                    ($request->comment ? 'Note: ' . $request->comment : ''),
                'assigned_to'    => $reviewer->id,
                'created_by'     => $reviewer->id,
                'parent_task_id' => $task->id,
                'planned_start'  => now()->toDateString(),
                'planned_end'    => $task->planned_end->isFuture() ? $task->planned_end->toDateString() : now()->addDays(3)->toDateString(),
                'priority'       => $task->priority,
                'status'         => 'pending',
            ]);

            TaskLog::create([
                'task_id' => $marketingTask->id,
                'user_id' => $reviewer->id,
                'action'  => 'created',
                'note'    => 'Auto-created as marketing processing task after approving: "' . $task->title . '"',
            ]);
        }

        // Notify the original task owner
        NotificationHelper::send(
            $task->assigned_to,
            'task_approved',
            'Your Task Was Approved!',
            '"' . $task->title . '" has been approved by ' . $reviewer->name,
            ['task_id' => $task->id, 'task_title' => $task->title]
        );

        // Notify all admins & super admins
        NotificationHelper::notifyAdmins(
            'task_approved',
            'Task Approved',
            '"' . $task->title . '" approved by ' . $reviewer->name,
            ['task_id' => $task->id, 'task_title' => $task->title],
            $reviewer->id
        );

        return back()->with('success', (!$isMarketingProcessingTask && !$isOwnTask)
            ? 'Task approved! A marketing processing task has been added to your task list.'
            : 'Task approved successfully.');
    }

    // Reject a task
    public function reject(Request $request, Task $task)
    {
        $request->validate([
            'comment' => 'required|string|max:500',
        ]);

        if ($task->status !== 'under_review') {
            return back()->with('error', 'Only tasks under review can be rejected.');
        }

        $task->update([
            'status'           => 'rejected',
            'rejection_reason' => $request->comment,
            'actual_end'       => null,
        ]);

        TaskApproval::create([
            'task_id'     => $task->id,
            'reviewed_by' => Auth::id(),
            'status'      => 'rejected',
            'comment'     => $request->comment,
        ]);

        TaskLog::create([
            'task_id' => $task->id,
            'user_id' => Auth::id(),
            'action'  => 'rejected',
            'note'    => 'Rejected: ' . $request->comment,
        ]);

        // Notify the task owner
        NotificationHelper::send(
            $task->assigned_to,
            'task_rejected',
            'Task Rejected',
            '"' . $task->title . '" was rejected: ' . $request->comment,
            ['task_id' => $task->id, 'task_title' => $task->title]
        );

        // Notify admins & super admins
        NotificationHelper::notifyAdmins(
            'task_rejected',
            'Task Rejected',
            '"' . $task->title . '" rejected by ' . Auth::user()->name . ': ' . $request->comment,
            ['task_id' => $task->id, 'task_title' => $task->title],
            Auth::id()
        );

        return back()->with('success', 'Task rejected. Employee has been notified.');
    }

    // Approval history
    public function history()
    {
        $approvals = TaskApproval::with(['task.assignedUser', 'reviewer'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('approvals.history', compact('approvals'));
    }
}
