<?php

namespace App\Http\Controllers;

use App\Helpers\NotificationHelper;
use App\Models\Task;
use App\Models\TaskLog;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class TaskController extends Controller
{
    // List all tasks (admin) or own tasks (employee)
    public function index(Request $request)
    {
        $user = Auth::user();
        $query = Task::with(['assignedUser', 'createdByUser']);

        // Employees see only their tasks
        if (!$user->canManageTasks() && !$user->isMarketing()) {
            $query->where('assigned_to', $user->id);
        }

        // Filters
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        if ($request->filled('priority')) {
            $query->where('priority', $request->priority);
        }
        if ($request->filled('assigned_to') && $user->canManageTasks()) {
            $query->where('assigned_to', $request->assigned_to);
        }
        if ($request->filled('search')) {
            $query->where('title', 'like', '%' . $request->search . '%');
        }

        $tasks = $query->orderBy('planned_end')->paginate(15)->withQueryString();
        $users = User::where('status', 'active')->orderBy('name')->get();

        return view('tasks.index', compact('tasks', 'users'));
    }

    // Show create form
    public function create()
    {
        $user  = Auth::user();
        $users = User::where('status', 'active')->orderBy('name')->get();

        // Designer / Developer can only assign to themselves
        if (!$user->canAssignTasks()) {
            $users = collect([$user]);
        }

        return view('tasks.create', compact('users'));
    }

    // Store new task
    public function store(Request $request)
    {
        $user = Auth::user();

        $validated = $request->validate([
            'title'         => 'required|string|max:200',
            'description'   => 'nullable|string',
            'assigned_to'   => 'required|exists:users,id',
            'planned_start' => 'required|date',
            'planned_end'   => 'required|date|after_or_equal:planned_start',
            'priority'      => 'required|in:low,medium,high',
            'attachment'    => 'nullable|file|max:5120',
        ]);

        // Designer / Developer can only assign to themselves
        if (!$user->canAssignTasks()) {
            $validated['assigned_to'] = $user->id;
        }

        $validated['created_by'] = $user->id;
        $validated['status']     = 'pending';

        if ($request->hasFile('attachment')) {
            $validated['attachment'] = $request->file('attachment')->store('task-attachments', 'public');
        }

        $task = Task::create($validated);

        // Log action
        TaskLog::create([
            'task_id' => $task->id,
            'user_id' => $user->id,
            'action'  => 'created',
            'note'    => 'Task created and assigned to ' . $task->assignedUser->name,
        ]);

        // Notify assigned user (if not self)
        if ($task->assigned_to !== $user->id) {
            NotificationHelper::send(
                $task->assigned_to,
                'task_assigned',
                'New Task Assigned',
                'You have a new task: "' . $task->title . '"',
                ['task_id' => $task->id, 'task_title' => $task->title]
            );
        }

        // Notify admins & super admins (exclude creator if they are admin)
        NotificationHelper::notifyAdmins(
            'task_assigned',
            'New Task Created',
            '"' . $task->title . '" assigned to ' . $task->assignedUser->name . ' by ' . $user->name,
            ['task_id' => $task->id, 'task_title' => $task->title],
            $user->id
        );

        return redirect()->route('tasks.index')->with('success', 'Task created successfully!');
    }

    // Show single task
    public function show(Task $task)
    {
        $user = Auth::user();

        // Employees can only see their own tasks
        if (!$user->canManageTasks() && !$user->isMarketing() && $task->assigned_to !== $user->id) {
            abort(403);
        }

        $task->load(['assignedUser', 'createdByUser', 'logs.user', 'approvals.reviewer', 'parentTask', 'marketingTask']);
        return view('tasks.show', compact('task'));
    }

    // Show edit form
    public function edit(Task $task)
    {
        $user = Auth::user();

        if (!$user->canManageTasks() && $task->assigned_to !== $user->id) {
            abort(403);
        }

        if (in_array($task->status, ['approved', 'completed'])) {
            return redirect()->route('tasks.show', $task)->with('error', 'Completed tasks cannot be edited.');
        }

        $users = User::where('status', 'active')->orderBy('name')->get();
        if (!$user->canAssignTasks()) {
            $users = collect([$user]);
        }

        return view('tasks.edit', compact('task', 'users'));
    }

    // Update task
    public function update(Request $request, Task $task)
    {
        $user = Auth::user();

        if (!$user->canManageTasks() && $task->assigned_to !== $user->id) {
            abort(403);
        }

        $validated = $request->validate([
            'title'         => 'required|string|max:200',
            'description'   => 'nullable|string',
            'assigned_to'   => 'required|exists:users,id',
            'planned_start' => 'required|date',
            'planned_end'   => 'required|date|after_or_equal:planned_start',
            'priority'      => 'required|in:low,medium,high',
            'attachment'    => 'nullable|file|max:5120',
        ]);

        if (!$user->canAssignTasks()) {
            $validated['assigned_to'] = $user->id;
        }

        if ($request->hasFile('attachment')) {
            if ($task->attachment) {
                Storage::disk('public')->delete($task->attachment);
            }
            $validated['attachment'] = $request->file('attachment')->store('task-attachments', 'public');
        }

        $task->update($validated);

        TaskLog::create([
            'task_id' => $task->id,
            'user_id' => $user->id,
            'action'  => 'updated',
            'note'    => 'Task details updated.',
        ]);

        return redirect()->route('tasks.show', $task)->with('success', 'Task updated successfully!');
    }

    // Start task (also handles re-start after rejection)
    public function start(Task $task)
    {
        $user = Auth::user();

        if ($task->assigned_to !== $user->id && !$user->canManageTasks()) {
            abort(403);
        }

        // Allow start from 'pending' OR re-start from 'rejected'
        if (!in_array($task->status, ['pending', 'rejected'])) {
            return back()->with('error', 'This task cannot be started at its current status.');
        }

        $isRestart = $task->status === 'rejected';

        $task->update([
            'status'           => 'in_progress',
            'actual_start'     => now(),
            'actual_end'       => null,       // reset end time on re-start
            'rejection_reason' => null,        // clear rejection reason
        ]);

        TaskLog::create([
            'task_id' => $task->id,
            'user_id' => $user->id,
            'action'  => 'started',
            'note'    => $isRestart
                ? 'Task re-started after rejection at ' . now()->format('d M Y, h:i A')
                : 'Task started at ' . now()->format('d M Y, h:i A'),
        ]);

        // Notify admins & super admins
        NotificationHelper::notifyAdmins(
            $isRestart ? 'task_restarted' : 'task_started',
            $isRestart ? 'Task Re-started' : 'Task Started',
            '"' . $task->title . '" ' . ($isRestart ? 're-started' : 'started') . ' by ' . $user->name,
            ['task_id' => $task->id, 'task_title' => $task->title],
            $user->id
        );

        return back()->with('success', $isRestart ? 'Task re-started successfully!' : 'Task started successfully!');
    }

    // Complete task (submit for review)
    public function complete(Request $request, Task $task)
    {
        $user = Auth::user();

        if ($task->assigned_to !== $user->id && !$user->canManageTasks()) {
            abort(403);
        }

        if ($task->status !== 'in_progress') {
            return back()->with('error', 'Only in-progress tasks can be marked as complete.');
        }

        $task->update([
            'status'     => 'under_review',
            'actual_end' => now(),
        ]);

        TaskLog::create([
            'task_id' => $task->id,
            'user_id' => $user->id,
            'action'  => 'submitted',
            'note'    => 'Task completed and submitted for marketing review at ' . now()->format('d M Y, h:i A'),
        ]);

        // Notify management (admin + super_admin + marketing) to review
        NotificationHelper::notifyManagement(
            'task_submitted',
            'Task Submitted for Review',
            '"' . $task->title . '" submitted for review by ' . $user->name,
            ['task_id' => $task->id, 'task_title' => $task->title],
            $user->id
        );

        return back()->with('success', 'Task submitted for review successfully!');
    }

    // Delete task (admin only)
    public function destroy(Task $task)
    {
        $user = Auth::user();

        if (!$user->canManageTasks()) {
            abort(403);
        }

        if ($task->attachment) {
            Storage::disk('public')->delete($task->attachment);
        }

        $task->delete();

        return redirect()->route('tasks.index')->with('success', 'Task deleted successfully!');
    }

    // Task history / logs (admin)
    public function history(Request $request)
    {
        $user = Auth::user();

        $query = TaskLog::with(['task', 'user'])->orderBy('created_at', 'desc');

        if (!$user->canManageTasks()) {
            $query->whereHas('task', function ($q) use ($user) {
                $q->where('assigned_to', $user->id);
            });
        }

        if ($request->filled('user_id') && $user->canManageTasks()) {
            $query->where('user_id', $request->user_id);
        }

        $logs = $query->paginate(20)->withQueryString();
        $users = User::where('status', 'active')->orderBy('name')->get();

        return view('tasks.history', compact('logs', 'users'));
    }
}
