<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Task extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'description',
        'assigned_to',
        'created_by',
        'parent_task_id',
        'planned_start',
        'planned_end',
        'actual_start',
        'actual_end',
        'status',
        'priority',
        'attachment',
        'rejection_reason',
    ];

    protected $casts = [
        'planned_start' => 'date',
        'planned_end'   => 'date',
        'actual_start'  => 'datetime',
        'actual_end'    => 'datetime',
    ];

    // Status badge color helper
    public function statusBadge(): string
    {
        return match($this->status) {
            'pending'      => 'secondary',
            'in_progress'  => 'primary',
            'under_review' => 'warning',
            'approved'     => 'success',
            'rejected'     => 'danger',
            'completed'    => 'dark',
            default        => 'secondary',
        };
    }

    // Priority badge color helper
    public function priorityBadge(): string
    {
        return match($this->priority) {
            'high'   => 'danger',
            'medium' => 'warning',
            'low'    => 'success',
            default  => 'secondary',
        };
    }

    // Check if task is overdue
    public function isOverdue(): bool
    {
        return $this->planned_end->isPast() && !in_array($this->status, ['completed', 'approved']);
    }

    // Relationships
    public function assignedUser()
    {
        return $this->belongsTo(User::class, 'assigned_to');
    }

    public function createdByUser()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function logs()
    {
        return $this->hasMany(TaskLog::class, 'task_id')->orderBy('created_at', 'desc');
    }

    public function approvals()
    {
        return $this->hasMany(TaskApproval::class, 'task_id')->orderBy('created_at', 'desc');
    }

    public function latestApproval()
    {
        return $this->hasOne(TaskApproval::class, 'task_id')->latestOfMany();
    }

    // Original task this was created from (for marketing tasks)
    public function parentTask()
    {
        return $this->belongsTo(Task::class, 'parent_task_id');
    }

    // Marketing task generated from this task after approval
    public function marketingTask()
    {
        return $this->hasOne(Task::class, 'parent_task_id');
    }
}
