<?php

use App\Http\Controllers\AuthController;
use App\Http\Controllers\DashboardController;
use App\Http\Controllers\NotificationController;
use App\Http\Controllers\ProfileController;
use App\Http\Controllers\TaskController;
use App\Http\Controllers\ApprovalController;
use App\Http\Controllers\UserController;
use Illuminate\Support\Facades\Route;

// Redirect root to dashboard or login
Route::get('/', function () {
    return auth()->check() ? redirect()->route('dashboard') : redirect()->route('login');
});

// ─── Authentication ───────────────────────────────────────────────
Route::middleware('guest')->group(function () {
    Route::get('/login', [AuthController::class, 'showLogin'])->name('login');
    Route::post('/login', [AuthController::class, 'login'])->name('login.submit');
});

Route::post('/logout', [AuthController::class, 'logout'])->name('logout')->middleware('auth');

// ─── Authenticated Routes ─────────────────────────────────────────
Route::middleware('auth')->group(function () {

    // Dashboard
    Route::get('/dashboard', [DashboardController::class, 'index'])->name('dashboard');

    // Profile (all users)
    Route::get('/profile',  [ProfileController::class, 'show'])->name('profile.show');
    Route::put('/profile',  [ProfileController::class, 'update'])->name('profile.update');

    // Notifications (all users)
    Route::get('/notifications',                            [NotificationController::class, 'index'])->name('notifications.index');
    Route::post('/notifications/read-all',                  [NotificationController::class, 'markAllRead'])->name('notifications.readAll');
    Route::post('/notifications/{notification}/read',        [NotificationController::class, 'markRead'])->name('notifications.read');

    // Tasks (all authenticated users)
    Route::get('/tasks',                [TaskController::class, 'index'])->name('tasks.index');
    Route::get('/tasks/create',         [TaskController::class, 'create'])->name('tasks.create');
    Route::post('/tasks',               [TaskController::class, 'store'])->name('tasks.store');
    Route::get('/tasks/history',        [TaskController::class, 'history'])->name('tasks.history');
    Route::get('/tasks/{task}',         [TaskController::class, 'show'])->name('tasks.show');
    Route::get('/tasks/{task}/edit',    [TaskController::class, 'edit'])->name('tasks.edit');
    Route::put('/tasks/{task}',         [TaskController::class, 'update'])->name('tasks.update');
    Route::patch('/tasks/{task}/start', [TaskController::class, 'start'])->name('tasks.start');
    Route::patch('/tasks/{task}/complete', [TaskController::class, 'complete'])->name('tasks.complete');
    Route::delete('/tasks/{task}',      [TaskController::class, 'destroy'])
         ->name('tasks.destroy')
         ->middleware('role:super_admin,admin');

    // Approvals (marketing, admin, super_admin)
    Route::middleware('role:super_admin,admin,marketing')->group(function () {
        Route::get('/approvals',                  [ApprovalController::class, 'index'])->name('approvals.index');
        Route::get('/approvals/history',          [ApprovalController::class, 'history'])->name('approvals.history');
        Route::post('/approvals/{task}/approve',  [ApprovalController::class, 'approve'])->name('approvals.approve');
        Route::post('/approvals/{task}/reject',   [ApprovalController::class, 'reject'])->name('approvals.reject');
    });

    // User Management (admin, super_admin)
    Route::middleware('role:super_admin,admin')->group(function () {
        Route::get('/users',            [UserController::class, 'index'])->name('users.index');
        Route::get('/users/create',     [UserController::class, 'create'])->name('users.create');
        Route::post('/users',           [UserController::class, 'store'])->name('users.store');
        Route::get('/users/{user}/edit',[UserController::class, 'edit'])->name('users.edit');
        Route::put('/users/{user}',     [UserController::class, 'update'])->name('users.update');
        Route::delete('/users/{user}',  [UserController::class, 'destroy'])
             ->name('users.destroy')
             ->middleware('role:super_admin');
    });
});
